/*
  libakai - C++ cross-platform akai sample disk reader
  Copyright (C) 2002-2003 Sbastien Mtrot

  Linux port by Christian Schoenebeck <cuse@users.sourceforge.net> 2003

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "DiskImage.h"

#ifdef _CARBON_
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>
#include <paths.h>
#include <sys/param.h>
#include <IOKit/IOKitLib.h>
#include <IOKit/IOBSD.h>
#include <IOKit/storage/IOMediaBSDClient.h>
#include <IOKit/storage/IOMedia.h>
#include <IOKit/storage/IOCDMedia.h>
#include <IOKit/storage/IOCDTypes.h>
#include <CoreFoundation/CoreFoundation.h>
#endif

//using namespace std;

DiskImage::DiskImage(char* path)
{
  Init();
  OpenStream(path);
}

#ifdef _CARBON_
kern_return_t FindEjectableCDMedia( io_iterator_t *mediaIterator )
{
  kern_return_t  kernResult;
  mach_port_t   masterPort;
  CFMutableDictionaryRef classesToMatch;

  kernResult = IOMasterPort( MACH_PORT_NULL, &masterPort );
  if ( KERN_SUCCESS != kernResult )
  {
    printf( "IOMasterPort returned %d\n", kernResult );
  }

  classesToMatch = IOServiceMatching( kIOCDMediaClass );
  if ( classesToMatch == NULL )
  {
      printf( "IOServiceMatching returned a NULL dictionary.\n" );
  }
  else
  {
    CFDictionarySetValue( classesToMatch, CFSTR( kIOMediaEjectableKey ), kCFBooleanTrue );
  }

  kernResult = IOServiceGetMatchingServices( masterPort, classesToMatch, mediaIterator );
  if ( KERN_SUCCESS != kernResult )
  {
      printf( "IOServiceGetMatchingServices returned %d\n", kernResult );
  }

  return kernResult;
}

kern_return_t GetBSDPath( io_iterator_t mediaIterator, char *bsdPath, CFIndex maxPathSize )
{
  io_object_t  nextMedia;
  kern_return_t kernResult = KERN_FAILURE;

  *bsdPath = '\0';

  nextMedia = IOIteratorNext( mediaIterator );
  if ( nextMedia )
  {
    CFTypeRef bsdPathAsCFString;

    bsdPathAsCFString = IORegistryEntryCreateCFProperty( nextMedia,
                                                         CFSTR( kIOBSDNameKey ), 
                                                         kCFAllocatorDefault,
                                                         0 );
    if ( bsdPathAsCFString )
    {
      size_t devPathLength;

      strcpy( bsdPath, _PATH_DEV );
      strcat( bsdPath, "r" );

      devPathLength = strlen( bsdPath );
      
      if ( CFStringGetCString( (__CFString*)bsdPathAsCFString,
                               bsdPath + devPathLength,
                               maxPathSize - devPathLength,
                               kCFStringEncodingASCII ) )
      {
          printf( "BSD path: %s\n", bsdPath );
          kernResult = KERN_SUCCESS;
      }

      CFRelease( bsdPathAsCFString );
    }
    IOObjectRelease( nextMedia );
  }
  
  return kernResult;
}

struct _CDMSF 
{
  u_char   minute;
  u_char   second;
  u_char   frame;
};

/* converting from minute-second to logical block entity */
#define MSF_TO_LBA(msf) (((((msf).minute * 60UL) + (msf).second) * 75UL) + (msf).frame - 150)

struct _CDTOC_Desc
{
  u_char        session;
  u_char        ctrl_adr; /* typed to be machine and compiler independent */
  u_char        tno;
  u_char        point;
  struct _CDMSF  address;
  u_char        zero;
  struct _CDMSF  p;
};

struct _CDTOC 
{
  u_short            length; /* in native cpu endian */
  u_char             first_session;
  u_char             last_session;
  struct _CDTOC_Desc  trackdesc[1];
};

static struct _CDTOC * ReadTOC (const char * devpath )
{ 
  struct _CDTOC * toc_p = NULL;
  io_iterator_t iterator = 0;
  io_registry_entry_t service = 0;
  CFDictionaryRef properties = 0;
  CFDataRef data = 0;
  mach_port_t port = 0; 
  char * devname;
  if (( devname = strrchr( devpath, '/' )) != NULL ) 
  {
    ++devname;
  } 
  else
  {
    devname = ( char *) devpath;
  }
  
  if ( IOMasterPort(bootstrap_port, &port ) != KERN_SUCCESS )
  {
    printf(  "IOMasterPort failed\n" ); goto Exit ;
  }
  if ( IOServiceGetMatchingServices( port, IOBSDNameMatching( port, 0, devname ),
                                     &iterator ) != KERN_SUCCESS ) 
  {
    printf( "IOServiceGetMatchingServices failed\n" ); goto Exit ;
  }

    char buffer[1024];
    IOObjectGetClass(iterator,buffer);
    printf("Service: %s\n",buffer);
     
  
  IOIteratorReset (iterator);
  service = IOIteratorNext( iterator );

  IOObjectRelease( iterator );

  iterator = 0; 
  while ( service && !IOObjectConformsTo( service, "IOCDMedia" ))
  { 
    char buffer[1024];
    IOObjectGetClass(service,buffer);
    printf("Service: %s\n",buffer);

    if ( IORegistryEntryGetParentIterator( service, kIOServicePlane, &iterator ) != KERN_SUCCESS ) 
    {
      printf( "IORegistryEntryGetParentIterator failed\n" );
      goto Exit ;
    }

    IOObjectRelease( service );
    service = IOIteratorNext( iterator );
    IOObjectRelease( iterator );

  }
  if ( service == NULL ) 
  {
    printf(  "CD media not found\n" ); goto Exit ;
  }
  if ( IORegistryEntryCreateCFProperties( service, (__CFDictionary **) &properties,
                                          kCFAllocatorDefault,
                                          kNilOptions ) != KERN_SUCCESS ) 
  {
    printf( "IORegistryEntryGetParentIterator failed\n" ); goto Exit ;
  }

  data = (CFDataRef) CFDictionaryGetValue( properties, CFSTR( "TOC" ) ); 
  if ( data == NULL ) 
  {
    printf(  "CFDictionaryGetValue failed\n" ); goto Exit ;
  } 
  else
  {

    CFRange range;
    CFIndex buflen;

    buflen = CFDataGetLength( data ) + 1;
    range = CFRangeMake( 0, buflen );
    toc_p = ( struct _CDTOC *) malloc( buflen );
    if ( toc_p == NULL )
    {
      printf( "Out of memory\n" ); goto Exit ;
    } 
    else 
    {
      CFDataGetBytes( data, range, ( unsigned char *) toc_p );
    } 

    /*
    fprintf( stderr, "Table of contents\n length %d first %d last %d\n",
            toc_p->length, toc_p->first_session, toc_p->last_session );
      */

    CFRelease( properties );

  }
Exit :
  if ( service )
  {
    IOObjectRelease( service );
  }

  return toc_p;
}
#endif // _CARBON_

DiskImage::DiskImage(int disk)
{
  Init();
#ifdef _WIN32_
  char buffer[1024];
  sprintf(buffer,"%c:\\",'A'+disk);
  mSize = GetFileSize(buffer,NULL);

  sprintf(buffer,"\\\\.\\%c:",'a'+disk);
  mFile = CreateFile(buffer, GENERIC_READ,0,NULL,OPEN_EXISTING, FILE_FLAG_RANDOM_ACCESS,NULL);

  DWORD junk;
  DISK_GEOMETRY dg;
  DISK_GEOMETRY* pdg = &dg;
  BOOL res = DeviceIoControl(mFile,
    IOCTL_DISK_GET_DRIVE_GEOMETRY,
    NULL, 0,
    pdg, sizeof(*pdg),
    &junk,
    NULL);

  if (res)
  {
    mSize = dg.BytesPerSector * dg.SectorsPerTrack * dg.TracksPerCylinder * dg.Cylinders.LowPart;
    mClusterSize = dg.BytesPerSector;
  }
#elif defined _CARBON_
  kern_return_t  kernResult;
  io_iterator_t mediaIterator;
  char  bsdPath[ MAXPATHLEN ];
  kernResult = FindEjectableCDMedia( &mediaIterator );
  kernResult = GetBSDPath( mediaIterator, bsdPath, sizeof( bsdPath ) );
  if ( bsdPath[ 0 ] != '\0' )
  {
		strcpy(bsdPath,"/dev/rdisk1s0");

      struct _CDTOC * toc = ReadTOC( bsdPath );
      if ( toc )
      {
        size_t toc_entries = ( toc->length - 2 ) / sizeof (struct _CDTOC_Desc );

        int start_sector = -1;
        int data_track = -1;
        // Iterate through the list backward. Pick the first data track and
        // get the address of the immediately previous (or following depending
        // on how you look at it).  The difference in the sector numbers
        // is returned as the sized of the data track.
        for (int i=toc_entries - 1; i>=0; i-- )
        {
          if ( start_sector != -1 )
          {
            start_sector = MSF_TO_LBA(toc->trackdesc[i].p) - start_sector;
            break ;

          }
          if (( toc->trackdesc[i].ctrl_adr >> 4) != 1 )
            continue ;
          if ( toc->trackdesc[i].ctrl_adr & 0x04 )
          {
            data_track = toc->trackdesc[i].point;
            start_sector = MSF_TO_LBA(toc->trackdesc[i].p);
          }
        }

        free( toc );
        if ( start_sector == -1 )
        {
          start_sector = 0;
        }
			}
//      mSize = start_sector;
//			printf("size %d\n",mSize);


    mFile = open(bsdPath,O_RDONLY);
    if (!mFile)
      printf("Error while opening file: %s\n",bsdPath);
    else
    {
      printf("opened file: %s\n",bsdPath);

			mSize = lseek(mFile,1000000000,SEEK_SET);
			printf("size %d\n",mSize);
			lseek(mFile,0,SEEK_SET);

			mSize = 700 * 1024 * 1024;

    }
  }
  if ( mediaIterator )
  {
    IOObjectRelease( mediaIterator );
  }
#elif LINUX
  OpenStream("/dev/cdrom");
#endif
}

void DiskImage::Init()
{
  mFile        = 0;
  mPos         = 0;
  mCluster     = (uint)-1;
  mStartFrame  = -1;
  mEndFrame    = -1;
#ifdef _WIN32_
  mpCache = (char*) VirtualAlloc(NULL,mClusterSize,MEM_COMMIT,PAGE_READWRITE);
#else
  mpCache = NULL; // we allocate the cache later when we know what type of media we access
#endif
}

DiskImage::~DiskImage()
{
#ifdef _WIN32_
  if (mFile != INVALID_HANDLE_VALUE)
  {
    CloseHandle(mFile);
  }
#elif defined _CARBON_ || LINUX
  if (mFile)
  {
    close(mFile);
  }
#endif
  if (mpCache)
  {
#ifdef _WIN32_
    VirtualFree(mpCache, 0, MEM_RELEASE);
#elif defined _CARBON_ || LINUX
    free(mpCache);
#endif
  }
}

akai_stream_state_t DiskImage::GetState() const
{
  if (!mFile)       return akai_stream_closed;
  if (mPos > mSize) return akai_stream_end_reached;
  return akai_stream_ready;
}

int DiskImage::GetPos() const
{
  return mPos;
}

int DiskImage::SetPos(int Where, akai_stream_whence_t Whence)
{
//  printf("setpos %d\n",Where);
  int w = Where;
  switch (Whence)
  {
  case akai_stream_start:
    mPos = w;
    break;
  /*case eStreamRewind:
    w = -w;*/
  case akai_stream_curpos:
    mPos += w;
    break;
  case akai_stream_end:
    mPos = mSize - w;
    break;
  }
//  if (mPos > mSize)
//    mPos = mSize;
  if (mPos < 0)
    mPos = 0;
  return mPos;
}

int DiskImage::Available (uint WordSize)
{
  return (mSize-mPos)/WordSize;
}

int DiskImage::Read(void* pData, uint WordCount, uint WordSize)
{
  int readbytes = 0;
  int sizetoread = WordCount * WordSize;

  while (sizetoread > 0) {
    if (mSize <= mPos) return readbytes / WordSize;
    int requestedCluster = (mRegularFile) ? mPos / mClusterSize
                                          : mPos / mClusterSize + mStartFrame;
    if (mCluster != requestedCluster) { // read the requested cluster into cache
      mCluster = requestedCluster;
#ifdef _WIN32_
      if (mCluster * mClusterSize != SetFilePointer(mFile, mCluster * mClusterSize, NULL, FILE_BEGIN)) {
		printf("ERROR: couldn't seek device!\n");
        if ((readbytes > 0) && (mEndian != eEndianNative)) {
          switch (WordSize) {
            case 2: bswap_16_s ((uint16*)pData, readbytes); break;
            case 4: bswap_32_s ((uint32*)pData, readbytes); break;
            case 8: bswap_64_s ((uint64*)pData, readbytes); break;
          }
        }
        return readbytes / WordSize;
      }
      DWORD size;
      ReadFile(mFile, mpCache, mClusterSize, &size, NULL);
#elif defined _CARBON_ || LINUX
      if (mCluster * mClusterSize != lseek(mFile, mCluster * mClusterSize, SEEK_SET))
        return readbytes / WordSize;
//			printf("trying to read %d bytes from device!\n",mClusterSize);
      int size = read(mFile, mpCache, mClusterSize);
//			printf("read %d bytes from device!\n",size);
#endif
    }
//		printf("read %d bytes at pos %d\n",WordCount*WordSize,mPos);

    int currentReadSize = sizetoread;
    int posInCluster = mPos % mClusterSize;
    if (currentReadSize > mClusterSize - posInCluster) // restrict to this current cached cluster.
      currentReadSize = mClusterSize - posInCluster;

    memcpy((uint8_t*)pData + readbytes, mpCache + posInCluster, currentReadSize);

    mPos       += currentReadSize;
    readbytes  += currentReadSize;
    sizetoread -= currentReadSize;
//		printf("new pos %d\n",mPos);
  }

#if 0
  if ((readbytes > 0) && (mEndian != eEndianNative))
    switch (WordSize)
    {
      case 2: bswap_16_s ((uint16_t*)pData, readbytes); break;
      case 4: bswap_32_s ((uint32_t*)pData, readbytes); break;
      case 8: bswap_64_s ((uint64_t*)pData, readbytes); break;
    }
#endif

  return readbytes / WordSize;
}

void DiskImage::ReadInt8(uint8_t* pData, uint WordCount) {
  Read(pData, WordCount, 1);
}

void DiskImage::ReadInt16(uint16_t* pData, uint WordCount) {
  int i;
  for (i = 0; i < WordCount; i++) {
    *(pData + i) = ReadInt16();
  }
}

void DiskImage::ReadInt32(uint32_t* pData, uint WordCount) {
  int i;
  for (i = 0; i < WordCount; i++) {
    *(pData + i) = ReadInt32();
  }
}

int DiskImage::ReadInt8(uint8_t* pData) {
  return Read(pData, 1, 1);
}

int DiskImage::ReadInt16(uint16_t* pData) {
  int result = Read(pData, 1, 2);
#if WORDS_BIGENDIAN
  swapBytes_16(pData);
#endif
  return result;
}

int DiskImage::ReadInt32(uint32_t* pData) {
  int result = Read(pData, 1, 4);
#if WORDS_BIGENDIAN
  swapBytes_32(pData);
#endif
  return result;
}

uint8_t DiskImage::ReadInt8()
{
  uint8_t word;
  Read(&word,1,1);
  return word;
}

uint16_t DiskImage::ReadInt16()
{
  uint16_t word;
  Read(&word,1,2);
#if WORDS_BIGENDIAN
  swapBytes_16(&word);
#endif
  return word;
}

uint32_t DiskImage::ReadInt32()
{
  uint32_t word;
  Read(&word,1,4);
#if WORDS_BIGENDIAN
  swapBytes_32(&word);
#endif
  return word;
}

void DiskImage::OpenStream(char* path) {
#ifdef _WIN32_
  mFile = CreateFile(path, GENERIC_READ,0,NULL,OPEN_EXISTING, FILE_FLAG_RANDOM_ACCESS,NULL);
  BY_HANDLE_FILE_INFORMATION FileInfo;
  GetFileInformationByHandle(mFile,&FileInfo);
  mSize = FileInfo.nFileSizeLow;
#elif _CARBON_ || LINUX
  struct stat filestat;
  stat(path,&filestat);
  mFile = open(path, O_RDONLY | O_NONBLOCK);
  if (mFile <= 0) {
    printf("Can't open %s\n", path);
    mFile = 0;
    return;
  }
  // TODO: we should also check here if 'path' is a link or something
  if (S_ISREG(filestat.st_mode)) { // regular file
    printf("Using regular Akai image file.\n");
    mRegularFile = true;
    mSize        = filestat.st_size;
    mClusterSize = DISK_CLUSTER_SIZE;
    mpCache = (char*) malloc(mClusterSize);
  }
  else { // CDROM
    mRegularFile = false;
    mClusterSize = CD_FRAMESIZE;
    mpCache = (char*) malloc(mClusterSize);

    struct cdrom_tochdr   tochdr;
    struct cdrom_tocentry tocentry;
    int prev_addr = 0;
    if (ioctl(mFile, CDROMREADTOCHDR, (unsigned long)&tochdr) < 0) {
      printf("Trying to read TOC of %s failed\n", path);
      close(mFile);
      mFile = 0;
      return;
    }
    printf("Total tracks: %d\n", tochdr.cdth_trk1);

    /* we access the CD with logical blocks as entity */
    tocentry.cdte_format = CDROM_LBA;

    int firstDataTrack = -1;
    int start, end, length;

    /* we pick up the lowest data track by iterating backwards, starting with Lead Out */
    for (int t = tochdr.cdth_trk1; t >= 0; t--) {
      tocentry.cdte_track = (t == tochdr.cdth_trk1) ? CDROM_LEADOUT : t + 1;
      if (ioctl(mFile, CDROMREADTOCENTRY, (unsigned long)&tocentry) < 0){
        printf("Failed to read TOC entry for track %d\n", tocentry.cdte_track);
        close(mFile);
        mFile = 0;
        return;
      }
      if (tocentry.cdte_track == CDROM_LEADOUT) {
          printf("Lead Out: Start(LBA)=%d\n", tocentry.cdte_addr.lba);
      }
      else {
        printf("Track %d: Start(LBA)=%d End(LBA)=%d Length(Blocks)=%d ",
               tocentry.cdte_track, tocentry.cdte_addr.lba, prev_addr - 1, prev_addr - tocentry.cdte_addr.lba);
        if (tocentry.cdte_ctrl & CDROM_DATA_TRACK) {
          printf("Type: Data\n");
          firstDataTrack = tocentry.cdte_track;
          start  = tocentry.cdte_addr.lba;
          end    = prev_addr - 1;
          length = prev_addr - tocentry.cdte_addr.lba;
        }
        else printf("Type: Audio\n");
      }
      prev_addr = tocentry.cdte_addr.lba;
    }

    if (firstDataTrack == -1) {
      printf("Sorry, no data track found on %s\n", path);
      close(mFile);
      mFile = 0;
      return;
    }

    printf("Ok, I'll pick track %d\n", firstDataTrack);
    mStartFrame = start;
    mEndFrame   = end;
    mSize       = length * CD_FRAMESIZE;
  } // CDROM
#endif
}

bool DiskImage::WriteImage(char* path)
{
#if _CARBON_ || LINUX
  const uint bufferSize = 524288; // 512 kB
  int fOut = open(path, O_WRONLY | O_NONBLOCK | O_CREAT | O_TRUNC);
  if (mFile <= 0) {
    printf("Can't open output file %s\n", path);
    return false;
  }
  uint8_t* pBuffer = new uint8_t[bufferSize];
  SetPos(0);
  while (Available() > 0) {
    int readBytes = Read(pBuffer,bufferSize,1);
    if (readBytes > 0) write(fOut,pBuffer,readBytes);
  }
  delete[] pBuffer;
  close(fOut);
  return true;
#endif // _CARBON_ || LINUX
}

inline void DiskImage::swapBytes_16(void* Word) {
  uint8_t byteCache;
  byteCache = *((uint8_t*) Word);
  *((uint8_t*) Word)     = *((uint8_t*) Word + 1);
  *((uint8_t*) Word + 1) = byteCache;
}

inline void DiskImage::swapBytes_32(void* Word) {
  uint8_t byteCache;
  byteCache = *((uint8_t*) Word);
  *((uint8_t*) Word)     = *((uint8_t*) Word + 3);
  *((uint8_t*) Word + 3) = byteCache;
  byteCache = *((uint8_t*) Word + 1);
  *((uint8_t*) Word + 1) = *((uint8_t*) Word + 2);
  *((uint8_t*) Word + 2) = byteCache;
}
