/*
  libakai - C++ cross-platform akai sample disk reader
  Copyright (C) 2002-2003 Sbastien Mtrot

  Linux port by Christian Schoenebeck <cuse@users.sourceforge.net> 2003

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
// AkaiDisk.cpp

#include "AkaiDisk.h"


//////////////////////////////////
// AkaiSample:
AkaiSample::AkaiSample(DiskImage* pDisk, AkaiVolume* pParent, const AkaiDirEntry& DirEntry)
  : AkaiDiskElement(pDisk->GetPos())
{
  mpParent = pParent;
  mpDisk = pDisk;
  mDirEntry = DirEntry;
  mpSamples = NULL;
  mHeaderOK = false;
  mPos = 0;

  //
  LoadHeader();
}

AkaiSample::~AkaiSample()
{
  if (mpSamples)
    free(mpSamples);
}

AkaiDirEntry AkaiSample::GetDirEntry()
{
  return mDirEntry;
}

bool AkaiSample::LoadSampleData()
{
  if (!LoadHeader())
    return false;
  if (mpSamples)
    return true;

  mpDisk->SetPos(mImageOffset);
  mpSamples = (int16_t*) malloc(mNumberOfSamples * sizeof(int16_t));
  if (!mpSamples)
    return false;

  mpDisk->ReadInt16((uint16_t*)mpSamples, mNumberOfSamples);
  return true;
}

void AkaiSample::ReleaseSampleData()
{
  if (!mpSamples)
    return;
  free(mpSamples);
  mpSamples = NULL;
}

int AkaiSample::SetPos(int Where, akai_stream_whence_t Whence)
{
  if (!mHeaderOK) return -1;
  int w = Where;
  switch (Whence)
  {
  case akai_stream_start:
    mPos = w;
    break;
  /*case eStreamRewind:
    w = -w;*/
  case akai_stream_curpos:
    mPos += w;
    break;
  case akai_stream_end:
    mPos = mNumberOfSamples - w;
    break;
  }
  if (mPos > mNumberOfSamples) mPos = mNumberOfSamples;
  if (mPos < 0) mPos = 0;
  return mPos;
}

int AkaiSample::Read(void* pBuffer, uint SampleCount)
{
  if (!mHeaderOK) return 0;

  if (mPos + SampleCount > mNumberOfSamples) SampleCount = mNumberOfSamples - mPos;

  mpDisk->SetPos(mImageOffset + mPos * 2); // FIXME: assumes 16 bit sample depth
  mpDisk->ReadInt16((uint16_t*)pBuffer, SampleCount);
  return SampleCount;
}

bool AkaiSample::LoadHeader()
{
  if (mHeaderOK)
    return true;

  mpDisk->SetPos(mpParent->GetParent()->GetOffset() + mDirEntry.mStart * AKAI_BLOCK_SIZE );

  // Length   Format      Description
  // --------------------------------------------------------------
  //    1                 3
  if (mpDisk->ReadInt8() != AKAI_SAMPLE_ID)
    return false;
  //    1                 Not important: 0 for 22050Hz, 1 for 44100Hz
  mpDisk->ReadInt8();
  //    1     unsigned    MIDI root note (C3=60)
  mMidiRootNote = mpDisk->ReadInt8();
  //   12     AKAII       Filename
  char buffer[13];
  mpDisk->Read(buffer, 12, 1);
  AkaiToAscii(buffer, 12);
  mName = buffer;

  //    1                 128
  mpDisk->ReadInt8();
  //    1     unsigned    Number of active loops
  mActiveLoops = mpDisk->ReadInt8();
  //    1     unsigned char       First active loop (0 for none)
  mFirstActiveLoop = mpDisk->ReadInt8();
  //    1                         0
  mpDisk->ReadInt8();
  //    1     unsigned    Loop mode: 0=in release 1=until release
  //                                 2=none       3=play to end
  mLoopMode = mpDisk->ReadInt8();
  //    1     signed      Cents tune -50...+50
  mTuneCents = mpDisk->ReadInt8();
  //    1     signed      Semi tune  -50...+50
  mTuneSemitones = mpDisk->ReadInt8();
  //    4                 0,8,2,0
  mpDisk->ReadInt8();
  mpDisk->ReadInt8();
  mpDisk->ReadInt8();
  mpDisk->ReadInt8();
  //
  //    4     unsigned    Number of sample words
  mNumberOfSamples = mpDisk->ReadInt32();
  //    4     unsigned    Start marker
  mStartMarker = mpDisk->ReadInt32();
  //    4     unsigned    End marker
  mEndMarker = mpDisk->ReadInt32();
  //
  //    4     unsigned    Loop 1 marker
  //    2     unsigned    Loop 1 fine length   (65536ths)
  //    4     unsigned    Loop 1 coarse length (words)
  //    2     unsigned    Loop 1 time          (msec. or 9999=infinite)
  //
  //   84     [as above]  Loops 2 to 8
  //
  int i;
  for (i=0; i<8; i++)
    mLoops[i].Load(mpDisk);
  //    4                 0,0,255,255
  mpDisk->ReadInt32();
  //    2     unsigned    Sampling frequency
  mSamplingFrequency = mpDisk->ReadInt16();
  //    1     signed char         Loop tune offset -50...+50
  mLoopTuneOffset = mpDisk->ReadInt8();

  mImageOffset = mpParent->GetParent()->GetOffset() + mDirEntry.mStart * AKAI_BLOCK_SIZE + 150; // 150 is the size of the header

  //FIXME: no header validation yet implemented
  return (mHeaderOK = true);
}

bool AkaiSampleLoop::Load(DiskImage* pDisk)
{
  //    4     unsigned    Loop 1 marker
  mMarker = pDisk->ReadInt32();
  //    2     unsigned    Loop 1 fine length   (65536ths)
  mFineLength = pDisk->ReadInt16();
  //    4     unsigned    Loop 1 coarse length (words)
  mCoarseLength = pDisk->ReadInt32();
  //    2     unsigned    Loop 1 time          (msec. or 9999=infinite)
  mTime = pDisk->ReadInt16();
  return true;
}

//////////////////////////////////
// AkaiProgram:
AkaiProgram::AkaiProgram(DiskImage* pDisk, AkaiVolume* pParent, const AkaiDirEntry& DirEntry)
  : AkaiDiskElement(pDisk->GetPos())
{
  mpParent = pParent; 
  mpDisk = pDisk;
  mDirEntry = DirEntry;
  mpKeygroups = NULL;
  Load();
}

AkaiProgram::~AkaiProgram()
{
  if (mpKeygroups)
    delete[] mpKeygroups;
}

AkaiDirEntry AkaiProgram::GetDirEntry()
{
  return mDirEntry;
}

bool AkaiProgram::Load()
{
  // Read each of the program parameters
  uint temppos = mpDisk->GetPos();
  mpDisk->SetPos(mpParent->GetParent()->GetOffset() + mDirEntry.mStart * AKAI_BLOCK_SIZE );
  //    byte     description                 default     range/comments
  //   ---------------------------------------------------------------------------
  //     1       program ID                  1
  uint8_t ProgID = mpDisk->ReadInt8();
  if (ProgID != AKAI_PROGRAM_ID)
  {
    mpDisk->SetPos(temppos);
    return false;
  }
  //     2-3     first keygroup address      150,0       
  uint16_t KeygroupAddress = mpDisk->ReadInt16();
  //     4-15    program name                10,10,10... AKAII character set
  char buffer[13];
  mpDisk->Read(buffer, 12, 1);
  AkaiToAscii(buffer, 12);
  mName = buffer;
  //     16      MIDI program number         0           0..127
  mMidiProgramNumber = mpDisk->ReadInt8();
  //     17      MIDI channel                0           0..15, 255=OMNI
  mMidiChannel = mpDisk->ReadInt8();
  //     18      polyphony                   15          1..16
  mPolyphony = mpDisk->ReadInt8();
  //     19      priority                    1           0=LOW 1=NORM 2=HIGH 3=HOLD
  mPriority = mpDisk->ReadInt8();
  //     20      low key                     24          24..127
  mLowKey = mpDisk->ReadInt8();
  //     21      high key                    127         24..127
  mHighKey = mpDisk->ReadInt8();
  //     22      octave shift                0           -2..2
  mOctaveShift = mpDisk->ReadInt8();
  //     23      aux output select           255         0..7, 255=OFF
  mAuxOutputSelect = mpDisk->ReadInt8();
  //     24      mix output level            99          0..99
  mMixOutputSelect = mpDisk->ReadInt8();
  //     25      mix output pan              0           -50..50
  mMixPan = mpDisk->ReadInt8();
  //     26      volume                      80          0..99
  mVolume = mpDisk->ReadInt8();
  //     27      vel>volume                  20          -50..50
  mVelocityToVolume = mpDisk->ReadInt8();
  //     28      key>volume                  0           -50..50
  mKeyToVolume = mpDisk->ReadInt8();
  //     29      pres>volume                 0           -50..50
  mPressureToVolume = mpDisk->ReadInt8();
  //     30      pan lfo rate                50          0..99
  mPanLFORate = mpDisk->ReadInt8();
  //     31      pan lfo depth               0           0..99
  mPanLFODepth = mpDisk->ReadInt8();
  //     32      pan lfo delay               0           0..99
  mPanLFODelay = mpDisk->ReadInt8();
  //     33      key>pan                     0           -50..50
  mKeyToPan = mpDisk->ReadInt8();
  //     34      lfo rate                    50          0..99
  mLFORate = mpDisk->ReadInt8();
  //     35      lfo depth                   0           0..99
  mLFODepth = mpDisk->ReadInt8();
  //     36      lfo delay                   0           0..99
  mLFODelay = mpDisk->ReadInt8();
  //     37      mod>lfo depth               30          0..99
  mModulationToLFODepth = mpDisk->ReadInt8();
  //     38      pres>lfo depth              0           0..99
  mPressureToLFODepth = mpDisk->ReadInt8();
  //     39      vel>lfo depth               0           0..99
  mVelocityToLFODepth = mpDisk->ReadInt8();
  //     40      bend>pitch                  2           0..12 semitones
  mBendToPitch = mpDisk->ReadInt8();
  //     41      pres>pitch                  0           -12..12 semitones
  mPressureToPitch = mpDisk->ReadInt8();
  //     42      keygroup crossfade          0           0=OFF 1=ON
  mKeygroupCrossfade = mpDisk->ReadInt8()?true:false;
  //     43      number of keygroups         1           1..99
  mNumberOfKeygroups = mpDisk->ReadInt8();
  //     44      (internal use)              0           program number
   mpDisk->ReadInt8();
  //     45-56   key temperament C,C#,D...   0           -25..25 cents
  uint i;
  for (i = 0; i<11; i++)
    mKeyTemperament[i] =  mpDisk->ReadInt8();
  //     57      fx output                   0           0=OFF 1=ON
  mFXOutput = mpDisk->ReadInt8()?true:false;
  //     58      mod>pan                     0           -50..50
  mModulationToPan = mpDisk->ReadInt8();
  //     59      stereo coherence            0           0=OFF 1=ON
  mStereoCoherence = mpDisk->ReadInt8()?true:false;
  //     60      lfo desync                  1           0=OFF 1=ON
  mLFODesync = mpDisk->ReadInt8()?true:false;
  //     61      pitch law                   0           0=LINEAR
  mPitchLaw = mpDisk->ReadInt8();
  //     62      voice re-assign             0           0=OLDEST 1=QUIETEST
  mVoiceReassign = mpDisk->ReadInt8();
  //     63      softped>volume              10          0..99
  mSoftpedToVolume = mpDisk->ReadInt8();
  //     64      softped>attack              10          0..99
  mSoftpedToAttack = mpDisk->ReadInt8();
  //     65      softped>filt                10          0..99
  mSoftpedToFilter = mpDisk->ReadInt8();
  //     66      tune cents                  0           -128..127 (-50..50 cents)
  mSoftpedToTuneCents = mpDisk->ReadInt8();
  //     67      tune semitones              0           -50..50
  mSoftpedToTuneSemitones = mpDisk->ReadInt8();
  //     68      key>lfo rate                0           -50..50
  mKeyToLFORate = mpDisk->ReadInt8();
  //     69      key>lfo depth               0           -50..50
  mKeyToLFODepth = mpDisk->ReadInt8();
  //     70      key>lfo delay               0           -50..50
  mKeyToLFODelay = mpDisk->ReadInt8();
  //     71      voice output scale          1           0=-6dB 1=0dB 2=+12dB
  mVoiceOutputScale = mpDisk->ReadInt8();
  //     72      stereo output scale         0           0=0dB 1=+6dB
  mStereoOutputScale = mpDisk->ReadInt8();
  //     73-150  (not used)

  // Read the key groups:
  if (mpKeygroups)
    delete[] mpKeygroups;
  mpKeygroups = new AkaiKeygroup[mNumberOfKeygroups];
  for (i = 0; i < mNumberOfKeygroups; i++)
  {
    // Go where the key group is on the disk:
    mpDisk->SetPos(mpParent->GetParent()->GetOffset() + mDirEntry.mStart * AKAI_BLOCK_SIZE + 150 * (i+1));
    if (!mpKeygroups[i].Load(mpDisk))
    {
      mpDisk->SetPos(temppos);
      return false;
    }
  }

  mpDisk->SetPos(temppos);
  return true;
}

uint AkaiProgram::ListSamples(std::list<String>& rSamples)
{
  return 0;
}

AkaiSample* AkaiProgram::GetSample(uint Index)
{
  return NULL;
}

AkaiSample* AkaiProgram::GetSample(const String& rName)
{
  return NULL;
}

// AkaiKeygroup:
bool AkaiKeygroup::Load(DiskImage* pDisk)
{
  uint i;
  //    byte     description                 default     range/comments
  //   ---------------------------------------------------------------------------
  //     1       keygroup ID                 2
  if (pDisk->ReadInt8() != AKAI_KEYGROUP_ID)
    return false;
  //     2-3     next keygroup address       44,1        300,450,600,750.. (16-bit)
  uint16_t NextKeygroupAddress = pDisk->ReadInt16();
  //     4       low key                     24          24..127
  mLowKey = pDisk->ReadInt8();
  //     5       high key                    127         24..127
  mHighKey = pDisk->ReadInt8();
  //     6       tune cents                  0           -128..127 (-50..50 cents)
  mTuneCents = pDisk->ReadInt8();
  //     7       tune semitones              0           -50..50
  mTuneSemitones = pDisk->ReadInt8();
  //     8       filter                      99          0..99
  mFilter = pDisk->ReadInt8();
  //     9       key>filter                  12          0..24 semitone/oct
  mKeyToFilter = pDisk->ReadInt8();
  //     10      vel>filt                    0           -50..50
  mVelocityToFilter = pDisk->ReadInt8();
  //     11      pres>filt                   0           -50..50
  mPressureToFilter = pDisk->ReadInt8();
  //     12      env2>filt                   0           -50..50
  mEnveloppe2ToFilter = pDisk->ReadInt8();

  //     13      env1 attack                 0           0..99
  //     14      env1 decay                  30          0..99
  //     15      env1 sustain                99          0..99
  //     16      env1 release                45          0..99
  //     17      env1 vel>attack             0           -50..50
  //     18      env1 vel>release            0           -50..50 
  //     19      env1 offvel>release         0           -50..50
  //     20      env1 key>dec&rel            0           -50..50
  //     21      env2 attack                 0           0..99
  //     22      env2 decay                  50          0..99
  //     23      env2 sustain                99          0..99
  //     24      env2 release                45          0..99
  //     25      env2 vel>attack             0           -50..50
  //     26      env2 vel>release            0           -50..50
  //     27      env2 offvel>release         0           -50..50
  //     28      env2 key>dec&rel            0           -50..50
  for (i=0; i<2; i++)
    mEnveloppes[i].Load(pDisk);

  //     29      vel>env2>filter             0           -50..50
  mVelocityToEnveloppe2ToFilter = pDisk->ReadInt8();
  //     30      env2>pitch                  0           -50..50
  mEnveloppe2ToPitch = pDisk->ReadInt8();
  //     31      vel zone crossfade          1           0=OFF 1=ON
  mVelocityZoneCrossfade = pDisk->ReadInt8()?true:false;
  //     32      vel zones used              4
  mVelocityZoneUsed = pDisk->ReadInt8();
  //     33      (internal use)              255
  pDisk->ReadInt8();
  //     34      (internal use)              255
  pDisk->ReadInt8();
  //

  //     35-46   sample 1 name               10,10,10... AKAII character set
  //     47      low vel                     0           0..127
  //     48      high vel                    127         0..127
  //     49      tune cents                  0           -128..127 (-50..50 cents)
  //     50      tune semitones              0           -50..50
  //     51      loudness                    0           -50..+50
  //     52      filter                      0           -50..+50
  //     53      pan                         0           -50..+50
  //     54      loop mode                   0           0=AS_SAMPLE 1=LOOP_IN_REL 
  //                                                     2=LOOP_UNTIL_REL 3=NO_LOOP 
  //                                                     4=PLAY_TO_END
  //     55      (internal use)              255
  //     56      (internal use)              255
  //     57-58   (internal use)              44,1
  //
  //     59-82   [repeat 35-58 for sample 2]
  //
  //     83-106  [repeat 35-58 for sample 3]
  //
  //     107-130 [repeat 35-58 for sample 4]
  // 
  for (i=0; i<4; i++)
    mSamples[i].Load(pDisk);
  
  //     131     beat detune                 0           -50..50
  mBeatDetune = pDisk->ReadInt8();
  //     132     hold attack until loop      0           0=OFF 1=ON
  mHoldAttackUntilLoop = pDisk->ReadInt8()?true:false;
  //     133-136 sample 1-4 key tracking     0           0=TRACK 1=FIXED
  for (i=0; i<4; i++)
    mSampleKeyTracking[i] = pDisk->ReadInt8()?true:false;
  //     137-140 sample 1-4 aux out offset   0           0..7
  for (i=0; i<4; i++)
    mSampleAuxOutOffset[i] = pDisk->ReadInt8();
  //     141-148 vel>sample start            0           -9999..9999 (16-bit signed)
  for (i=0; i<4; i++)
    mVelocityToSampleStart[i] = pDisk->ReadInt8();
  //     149     vel>volume offset           0           -50..50
  for (i=0; i<4; i++)
    mVelocityToVolumeOffset[i] = pDisk->ReadInt8();
  //     150     (not used)

  return true;
}

bool AkaiEnveloppe::Load(DiskImage* pDisk)
{
  //     13      env1 attack                 0           0..99
  mAttack = pDisk->ReadInt8();
  //     14      env1 decay                  30          0..99
  mDecay = pDisk->ReadInt8();
  //     15      env1 sustain                99          0..99
  mSustain = pDisk->ReadInt8();
  //     16      env1 release                45          0..99
  mRelease = pDisk->ReadInt8();
  //     17      env1 vel>attack             0           -50..50
  mVelocityToAttack = pDisk->ReadInt8();
  //     18      env1 vel>release            0           -50..50 
  mVelocityToRelease = pDisk->ReadInt8();
  //     19      env1 offvel>release         0           -50..50
  mOffVelocityToRelease = pDisk->ReadInt8();
  //     20      env1 key>dec&rel            0           -50..50
  mKeyToDecayAndRelease = pDisk->ReadInt8();
  return true;
}

bool AkaiKeygroupSample::Load(DiskImage* pDisk)
{
  //     35-46   sample 1 name               10,10,10... AKAII character set
  char buffer[13];
  pDisk->Read(buffer, 12, 1);
  AkaiToAscii(buffer, 12);
  mName = buffer;

  //     47      low vel                     0           0..127
  mLowLevel = pDisk->ReadInt8();
  //     48      high vel                    127         0..127
  uint8_t mHighLevel = pDisk->ReadInt8();
  //     49      tune cents                  0           -128..127 (-50..50 cents)
  int8_t mTuneCents = pDisk->ReadInt8();
  //     50      tune semitones              0           -50..50
  int8_t mTuneSemitones = pDisk->ReadInt8();
  //     51      loudness                    0           -50..+50
  int8_t mLoudness = pDisk->ReadInt8();
  //     52      filter                      0           -50..+50
  int8_t mFilter = pDisk->ReadInt8();
  //     53      pan                         0           -50..+50
  int8_t mPan = pDisk->ReadInt8();
  //     54      loop mode                   0           0=AS_SAMPLE 1=LOOP_IN_REL 
  //                                                     2=LOOP_UNTIL_REL 3=NO_LOOP 
  //                                                     4=PLAY_TO_END
  uint8_t mLoopMode = pDisk->ReadInt8();
  //     55      (internal use)              255
  pDisk->ReadInt8();
  //     56      (internal use)              255
  pDisk->ReadInt8();
  //     57-58   (internal use)              44,1
  pDisk->ReadInt16();
  //

  return true;
}

//////////////////////////////////
// AkaiVolume:
AkaiVolume::AkaiVolume(DiskImage* pDisk, AkaiPartition* pParent, const AkaiDirEntry& DirEntry)
  : AkaiDiskElement()
{
  mpDisk = pDisk;
  mpParent = pParent;
  mDirEntry = DirEntry;

  if (mDirEntry.mType != AKAI_TYPE_DIR_S1000 && mDirEntry.mType != AKAI_TYPE_DIR_S3000)
  {
    printf("Creating Unknown Volume type! %d\n",mDirEntry.mType);
#ifdef _WIN32_
    OutputDebugString("Creating Unknown Volume type!\n");
#endif
  }
}

AkaiVolume::~AkaiVolume()
{
  {
    std::list<AkaiProgram*>::iterator it;
    std::list<AkaiProgram*>::iterator end = mpPrograms.end();
    for (it = mpPrograms.begin(); it != end; it++)
      if (*it)
        (*it)->Release();
  }

  {
    std::list<AkaiSample*>::iterator it;
    std::list<AkaiSample*>::iterator end = mpSamples.end();
    for (it = mpSamples.begin(); it != end; it++)
      if (*it)
        (*it)->Release();
  }
}

uint AkaiVolume::ReadDir()
{
  uint i;
  if (mpPrograms.empty())
  {
    uint maxfiles = ReadFAT(mpDisk, mpParent,mDirEntry.mStart) ? AKAI_MAX_FILE_ENTRIES_S1000 : AKAI_MAX_FILE_ENTRIES_S3000;
    for (i = 0; i < maxfiles; i++) 
    {
      AkaiDirEntry DirEntry;
      ReadDirEntry(mpDisk, mpParent, DirEntry, mDirEntry.mStart, i); 
      DirEntry.mIndex = i;
      if (DirEntry.mType == 'p') 
      {
        AkaiProgram* pProgram = new AkaiProgram(mpDisk, this, DirEntry);
        pProgram->Acquire();
        mpPrograms.push_back(pProgram);
      }
      else if (DirEntry.mType == 's') 
      {
        AkaiSample* pSample = new AkaiSample(mpDisk, this, DirEntry);
        pSample->Acquire();
        mpSamples.push_back(pSample);
      }
    }
  }
  return (uint)(mpPrograms.size() + mpSamples.size());
}

uint AkaiVolume::ListPrograms(std::list<AkaiDirEntry>& rPrograms)
{
  rPrograms.clear();
  ReadDir();

  std::list<AkaiProgram*>::iterator it;
  std::list<AkaiProgram*>::iterator end = mpPrograms.end();
  for (it = mpPrograms.begin(); it != end; it++)
    if (*it)
      rPrograms.push_back((*it)->GetDirEntry());
  return (uint)rPrograms.size();
}

AkaiProgram* AkaiVolume::GetProgram(uint Index)
{
  uint i = 0;

  if (mpPrograms.empty())
  {
    std::list<AkaiDirEntry> dummy;
    ListPrograms(dummy);
  }

  std::list<AkaiProgram*>::iterator it;
  std::list<AkaiProgram*>::iterator end = mpPrograms.end();
  for (it = mpPrograms.begin(); it != end; it++)
  {
    if (*it && i == Index)
    {
      (*it)->Acquire();
      return *it;
    }
    i++;
  }
  return NULL;
}

AkaiProgram* AkaiVolume::GetProgram(const String& rName)
{
  if (mpPrograms.empty())
  {
    std::list<AkaiDirEntry> dummy;
    ListPrograms(dummy);
  }

  std::list<AkaiProgram*>::iterator it;
  std::list<AkaiProgram*>::iterator end = mpPrograms.end();
  for (it = mpPrograms.begin(); it != end; it++)
  {
    if (*it && rName == (*it)->GetDirEntry().mName)
    {
      (*it)->Acquire();
      return *it;
    }
  }
  return NULL;
}

uint AkaiVolume::ListSamples(std::list<AkaiDirEntry>& rSamples)
{
  rSamples.clear();
  ReadDir();

  std::list<AkaiSample*>::iterator it;
  std::list<AkaiSample*>::iterator end = mpSamples.end();
  for (it = mpSamples.begin(); it != end; it++)
    if (*it)
      rSamples.push_back((*it)->GetDirEntry());
  return (uint)rSamples.size();
}

AkaiSample* AkaiVolume::GetSample(uint Index)
{
  uint i = 0;

  if (mpSamples.empty())
  {
    std::list<AkaiDirEntry> dummy;
    ListSamples(dummy);
  }

  std::list<AkaiSample*>::iterator it;
  std::list<AkaiSample*>::iterator end = mpSamples.end();
  for (it = mpSamples.begin(); it != end; it++)
  {
    if (*it && i == Index)
    {
      (*it)->Acquire();
      return *it;
    }
    i++;
  }
  return NULL;
}

AkaiSample* AkaiVolume::GetSample(const String& rName)
{
  if (mpSamples.empty())
  {
    std::list<AkaiDirEntry> dummy;
    ListSamples(dummy);
  }

  std::list<AkaiSample*>::iterator it;
  std::list<AkaiSample*>::iterator end = mpSamples.end();
  for (it = mpSamples.begin(); it != end; it++)
  {
    if (*it && rName == (*it)->GetDirEntry().mName)
    {
      (*it)->Acquire();
      return *it;
    }
  }
  return NULL;
}

AkaiDirEntry AkaiVolume::GetDirEntry()
{
  return mDirEntry;
}

bool AkaiVolume::IsEmpty()
{
  return ReadDir() == 0;
}


//////////////////////////////////
// AkaiPartition:
AkaiPartition::AkaiPartition(DiskImage* pDisk, AkaiDisk* pParent)
{
  mpDisk = pDisk;
  mpParent = pParent;
}

AkaiPartition::~AkaiPartition()
{
  std::list<AkaiVolume*>::iterator it;
  std::list<AkaiVolume*>::iterator end = mpVolumes.end();
  for (it = mpVolumes.begin(); it != end; it++)
    if (*it)
      (*it)->Release();
}

uint AkaiPartition::ListVolumes(std::list<AkaiDirEntry>& rVolumes)
{
  rVolumes.clear();
  uint i;
  if (mpVolumes.empty())
  {
    for (i = 0; i < AKAI_MAX_DIR_ENTRIES; i++)
    {
      AkaiDirEntry DirEntry;
      ReadDirEntry(mpDisk, this, DirEntry, AKAI_ROOT_ENTRY_OFFSET, i);
      DirEntry.mIndex = i;
      if (DirEntry.mType == AKAI_TYPE_DIR_S1000 || DirEntry.mType == AKAI_TYPE_DIR_S3000)
      {
        AkaiVolume* pVolume = new AkaiVolume(mpDisk, this, DirEntry);
        pVolume->Acquire();
        if (!pVolume->IsEmpty())
        {
          mpVolumes.push_back(pVolume);
          rVolumes.push_back(DirEntry);
        }
        else
          pVolume->Release();
      }
    }
  }
  else
  {
    std::list<AkaiVolume*>::iterator it;
    std::list<AkaiVolume*>::iterator end = mpVolumes.end();
    for (it = mpVolumes.begin(); it != end; it++)
      if (*it)
        rVolumes.push_back((*it)->GetDirEntry());
  }
  return (uint)rVolumes.size();
}

AkaiVolume* AkaiPartition::GetVolume(uint Index)
{
  uint i = 0;

  if (mpVolumes.empty())
  {
    std::list<AkaiDirEntry> dummy;
    ListVolumes(dummy);
  }

  std::list<AkaiVolume*>::iterator it;
  std::list<AkaiVolume*>::iterator end = mpVolumes.end();
  for (it = mpVolumes.begin(); it != end; it++)
  {
    if (*it && i == Index)
    {
      (*it)->Acquire();
      return *it;
    }
    i++;
  }
  return NULL;
}

AkaiVolume* AkaiPartition::GetVolume(const String& rName)
{
  if (mpVolumes.empty())
  {
    std::list<AkaiDirEntry> dummy;
    ListVolumes(dummy);
  }

  std::list<AkaiVolume*>::iterator it;
  std::list<AkaiVolume*>::iterator end = mpVolumes.end();
  for (it = mpVolumes.begin(); it != end; it++)
  {
    if (*it && rName == (*it)->GetDirEntry().mName)
    {
      (*it)->Acquire();
      return *it;
    }
  }
  return NULL;
}

bool AkaiPartition::IsEmpty()
{
  std::list<AkaiDirEntry> Volumes;
  return ListVolumes(Volumes) == 0;
}


//////////////////////////////////
// AkaiDisk:
AkaiDisk::AkaiDisk(DiskImage* pDisk)
{
  mpDisk = pDisk;
}

AkaiDisk::~AkaiDisk()
{
  std::list<AkaiPartition*>::iterator it;
  std::list<AkaiPartition*>::iterator end = mpPartitions.end();
  for (it = mpPartitions.begin(); it != end ; it++)
    if (*it)
      (*it)->Release();
}

uint AkaiDisk::GetPartitionCount()
{
  if (!mpPartitions.empty())
    return (uint)mpPartitions.size();

  uint offset = 0;
  uint16_t size = 0;
  while (size != AKAI_PARTITION_END_MARK && size != 0x0fff && size != 0xffff
         && size<30720 && mpPartitions.size()<9)
  {
    // printf("size: %x\t",size);
    AkaiPartition* pPartition = new AkaiPartition(mpDisk,this);
    pPartition->Acquire();
    pPartition->SetOffset(offset);

    if (!pPartition->IsEmpty())
    {
      mpPartitions.push_back(pPartition); // Add this partitions' offset to the list.
    }

    mpDisk->SetPos(offset);
    if (!mpDisk->ReadInt16(&size))
      return (uint)mpPartitions.size();
    uint t = size;
    offset +=  AKAI_BLOCK_SIZE * t;
//		printf("new offset %d / size %d\n",offset,size);
  }

  return (uint)mpPartitions.size();
}

AkaiPartition* AkaiDisk::GetPartition(uint count)
{
  std::list<AkaiPartition*>::iterator it;
  std::list<AkaiPartition*>::iterator end = mpPartitions.end();
  uint i = 0;
  for (i = 0, it = mpPartitions.begin(); it != end && i != count; i++) it++;

  if (i != count || it == end)
    return NULL;

  (*it)->Acquire();
  return *it;
}

/////////////////////////
// AkaiDiskElement

int AkaiDiskElement::ReadFAT(DiskImage* pDisk, AkaiPartition* pPartition, int block)
{ 
  int16_t value = 0;
  pDisk->SetPos(pPartition->GetOffset()+AKAI_FAT_OFFSET + block*2); 
  pDisk->Read(&value, 2,1); 
  return value;
}


bool AkaiDiskElement::ReadDirEntry(DiskImage* pDisk, AkaiPartition* pPartition, AkaiDirEntry& rEntry, int block, int pos)
{
  char buffer[13];

  if (block == AKAI_ROOT_ENTRY_OFFSET)
  {
    pDisk->SetPos(pPartition->GetOffset()+AKAI_DIR_ENTRY_OFFSET + pos * AKAI_DIR_ENTRY_SIZE);
    pDisk->Read(buffer, 12, 1);
    AkaiToAscii(buffer, 12);
    rEntry.mName = buffer;

    pDisk->ReadInt16(&rEntry.mType);
    pDisk->ReadInt16(&rEntry.mStart);
    rEntry.mSize = 0;
    return true;
  }
  else
  {
    if (pos < 341)
    {
      pDisk->SetPos(block * AKAI_BLOCK_SIZE + pos * AKAI_FILE_ENTRY_SIZE + pPartition->GetOffset());
    }
    else 
    {
      int temp; 
      temp = ReadFAT(pDisk, pPartition, block);
      pDisk->SetPos(pPartition->GetOffset()+temp * AKAI_BLOCK_SIZE + (pos - 341) * AKAI_FILE_ENTRY_SIZE);
    }

    pDisk->Read(buffer, 12, 1);
    AkaiToAscii(buffer, 12);
    rEntry.mName = buffer; 

    uint8_t t1,t2,t3;
    pDisk->SetPos(4,akai_stream_curpos);
    pDisk->Read(&t1, 1,1);
    rEntry.mType = t1; 

    pDisk->Read(&t1, 1,1);
    pDisk->Read(&t2, 1,1);
    pDisk->Read(&t3, 1,1);
    rEntry.mSize = (unsigned char)t1 | ((unsigned char)t2 <<8) | ((unsigned char)t3<<16); 

    pDisk->ReadInt16(&rEntry.mStart,1);
    return true; 
  } // not root block
}

void AkaiDiskElement::AkaiToAscii(char * buffer, int length) 
{ 
  int i;

  for (i = 0; i < length; i++)
  {
    if (buffer[i]>=0 && buffer[i]<=9)
      buffer[i] +=48;
    else if (buffer[i]==10)
      buffer[i] = 32;
    else if (buffer[i]>=11 && buffer[i]<=36)
      buffer[i] = 64+(buffer[i]-10);
    else
      buffer[i] = 32;
  }
  buffer[length] = '\0';
  while (length-- > 0 && buffer[length] == 32)
  {
    // This block intentionaly left blank :)
  }
  buffer[length+1] = '\0';
}
