/*
  libakai - C++ cross-platform akai sample disk reader
  Copyright (C) 2002-2003 Sbastien Mtrot

  Linux port by Christian Schoenebeck <cuse@users.sourceforge.net> 2003

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
// DiskImage.h

#ifndef __DiskImage_h__
#define __DiskImage_h__

// for use with autoconf
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdint.h>
#include <string>

#if LINUX
# include <stdio.h>
# include <stdlib.h>
# include <sys/types.h>
# include <sys/stat.h>
# include <sys/fcntl.h>
# include <sys/ioctl.h>
# include <unistd.h>
# include <iostream>
# include <linux/cdrom.h>
#endif

/* current state of the Akai stream */
typedef enum {
  akai_stream_ready       = 0,
  akai_stream_end_reached = 1,
  akai_stream_closed      = 2
} akai_stream_state_t;

/* stream position dependent to these relations */
typedef enum {
  akai_stream_start  = 0,
  akai_stream_curpos = 1,
  akai_stream_end    = 2
} akai_stream_whence_t;


/* We need to cache IO access to reduce IO system calls which else would slow
   down things tremendously. For that we differ between the following two
   cache sizes:

   CD_FRAMESIZE       for CDROM access
   DISK_CLUSTER_SIZE  for normal IO access (e.g. from hard disk)

   Not yet sure if these are the optimum sizes.
 */

#ifndef CD_FRAMESIZE
#  define CD_FRAMESIZE 2048 /* frame size for Yellow Book, Form 1 */
#endif

#define DISK_CLUSTER_SIZE 61440 /* 60 kB */


typedef std::string String;

class DiskImage
{
public:
  DiskImage(char* path); ///< Open an image from a file.
  DiskImage(int disk); ///< Open an image from a device number (0='a:', 1='b:', etc...).

  bool WriteImage(char* path); ///< Extract Akai data track and write it into a regular file.

  virtual ~DiskImage();

  virtual akai_stream_state_t GetState() const;
  virtual int GetPos() const;
  virtual int SetPos(int Where, akai_stream_whence_t Whence = akai_stream_start);

  virtual int Available (uint WordSize = 1);
  virtual int Read (void* pData, uint WordCount, uint WordSize); ///< Returns number of successfully read words.

  void     ReadInt8(uint8_t* pData, uint WordCount);
  void     ReadInt16(uint16_t* pData, uint WordCount);
  void     ReadInt32(uint32_t* pData, uint WordCount);
  int      ReadInt8(uint8_t* pData);    ///< Returns number of successfully read 8 Bit words.
  int      ReadInt16(uint16_t* pData);  ///< Returns number of successfully read 16 Bit words.
  int      ReadInt32(uint32_t* pData);  ///< Returns number of successfully read 32 Bit words.
  uint8_t  ReadInt8();
  uint16_t ReadInt16();
  uint32_t ReadInt32();


  virtual uint GetError() const
  {
    return 0;
  }

  /*virtual const nglChar* GetErrorStr(uint err) const
  {
    return _T("No Error");
  }*/

protected:
#ifdef _WIN32_
  HANDLE mFile;
#elif defined _CARBON_ || LINUX
  int mFile;
#endif
  bool mRegularFile;
  int mPos;
  int mCluster;
  int mClusterSize;
  int mSize; /* in bytes */
#if LINUX
  /* start and end of the data track we chose (if we're reading from CDROM) */
  int mStartFrame;
  int mEndFrame;
#endif // LINUX
  char* mpCache;

  void OpenStream(char* path);
  inline void swapBytes_16(void* Word);
  inline void swapBytes_32(void* Word);

private:
  void Init();
};

#endif
